/**************************************************************************
 *
 * Copyright 2012-2021 VMware, Inc.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDERS, AUTHORS AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 **************************************************************************/

/*
 * Format.h --
 *    Functions for format manipulation.
 */


#include <stdint.h>

#include "Format.h"
#include "State.h"

#include "Debug.h"


/* XXX: for a time llvmpipe didn't support SRGB */
#define HAVE_SRGB 1


enum pipe_format
FormatTranslate(DXGI_FORMAT Format, BOOL depth)
{
   switch (Format) {

   case DXGI_FORMAT_UNKNOWN:
      return PIPE_FORMAT_NONE;

   case DXGI_FORMAT_R32G32B32A32_FLOAT:
      return PIPE_FORMAT_R32G32B32A32_FLOAT;
   case DXGI_FORMAT_R32G32B32A32_UINT:
      return PIPE_FORMAT_R32G32B32A32_UINT;
   case DXGI_FORMAT_R32G32B32A32_SINT:
      return PIPE_FORMAT_R32G32B32A32_SINT;
   case DXGI_FORMAT_R32G32B32_FLOAT:
      return PIPE_FORMAT_R32G32B32_FLOAT;
   case DXGI_FORMAT_R32G32B32_UINT:
      return PIPE_FORMAT_R32G32B32_UINT;
   case DXGI_FORMAT_R32G32B32_SINT:
      return PIPE_FORMAT_R32G32B32_SINT;
   case DXGI_FORMAT_R16G16B16A16_UNORM:
      return PIPE_FORMAT_R16G16B16A16_UNORM;
   case DXGI_FORMAT_R16G16B16A16_UINT:
      return PIPE_FORMAT_R16G16B16A16_UINT;
   case DXGI_FORMAT_R16G16B16A16_SNORM:
      return PIPE_FORMAT_R16G16B16A16_SNORM;
   case DXGI_FORMAT_R16G16B16A16_SINT:
      return PIPE_FORMAT_R16G16B16A16_SINT;
   case DXGI_FORMAT_R32G32_FLOAT:
      return PIPE_FORMAT_R32G32_FLOAT;
   case DXGI_FORMAT_R32G32_UINT:
      return PIPE_FORMAT_R32G32_UINT;
   case DXGI_FORMAT_R32G32_SINT:
      return PIPE_FORMAT_R32G32_SINT;
   case DXGI_FORMAT_R8G8B8A8_UNORM:
      return PIPE_FORMAT_R8G8B8A8_UNORM;
   case DXGI_FORMAT_R8G8B8A8_UINT:
      return PIPE_FORMAT_R8G8B8A8_UINT;
   case DXGI_FORMAT_R8G8B8A8_SNORM:
      return PIPE_FORMAT_R8G8B8A8_SNORM;
   case DXGI_FORMAT_R8G8B8A8_SINT:
      return PIPE_FORMAT_R8G8B8A8_SINT;
   case DXGI_FORMAT_R16G16_UNORM:
      return PIPE_FORMAT_R16G16_UNORM;
   case DXGI_FORMAT_R16G16_UINT:
      return PIPE_FORMAT_R16G16_UINT;
   case DXGI_FORMAT_R16G16_SNORM:
      return PIPE_FORMAT_R16G16_SNORM;
   case DXGI_FORMAT_R16G16_SINT:
      return PIPE_FORMAT_R16G16_SINT;
   case DXGI_FORMAT_D32_FLOAT:
      return PIPE_FORMAT_Z32_FLOAT;
   case DXGI_FORMAT_R32_FLOAT:
      return PIPE_FORMAT_R32_FLOAT;
   case DXGI_FORMAT_R32_UINT:
      return PIPE_FORMAT_R32_UINT;
   case DXGI_FORMAT_R32_SINT:
      return PIPE_FORMAT_R32_SINT;
   case DXGI_FORMAT_D24_UNORM_S8_UINT:
      return PIPE_FORMAT_Z24_UNORM_S8_UINT;
   case DXGI_FORMAT_R8G8_UNORM:
      return PIPE_FORMAT_R8G8_UNORM;
   case DXGI_FORMAT_R8G8_UINT:
      return PIPE_FORMAT_R8G8_UINT;
   case DXGI_FORMAT_R8G8_SNORM:
      return PIPE_FORMAT_R8G8_SNORM;
   case DXGI_FORMAT_R8G8_SINT:
      return PIPE_FORMAT_R8G8_SINT;
   case DXGI_FORMAT_D16_UNORM:
      return PIPE_FORMAT_Z16_UNORM;
   case DXGI_FORMAT_R16_UNORM:
      return PIPE_FORMAT_R16_UNORM;
   case DXGI_FORMAT_R16_UINT:
      return PIPE_FORMAT_R16_UINT;
   case DXGI_FORMAT_R16_SNORM:
      return PIPE_FORMAT_R16_SNORM;
   case DXGI_FORMAT_R16_SINT:
      return PIPE_FORMAT_R16_SINT;
   case DXGI_FORMAT_R8_UNORM:
      return PIPE_FORMAT_R8_UNORM;
   case DXGI_FORMAT_R8_UINT:
      return PIPE_FORMAT_R8_UINT;
   case DXGI_FORMAT_R8_SNORM:
      return PIPE_FORMAT_R8_SNORM;
   case DXGI_FORMAT_R8_SINT:
      return PIPE_FORMAT_R8_SINT;
   case DXGI_FORMAT_A8_UNORM:
      return PIPE_FORMAT_A8_UNORM;
   case DXGI_FORMAT_BC1_UNORM:
      return PIPE_FORMAT_DXT1_RGBA;
   case DXGI_FORMAT_BC1_UNORM_SRGB:
#if HAVE_SRGB
      return PIPE_FORMAT_DXT1_SRGBA;
#else
      return PIPE_FORMAT_DXT1_RGBA;
#endif
   case DXGI_FORMAT_BC2_UNORM:
      return PIPE_FORMAT_DXT3_RGBA;
   case DXGI_FORMAT_BC2_UNORM_SRGB:
#if HAVE_SRGB
      return PIPE_FORMAT_DXT3_SRGBA;
#else
      return PIPE_FORMAT_DXT3_RGBA;
#endif
   case DXGI_FORMAT_BC3_UNORM:
      return PIPE_FORMAT_DXT5_RGBA;
   case DXGI_FORMAT_BC3_UNORM_SRGB:
#if HAVE_SRGB
      return PIPE_FORMAT_DXT5_SRGBA;
#else
      return PIPE_FORMAT_DXT5_RGBA;
#endif
   case DXGI_FORMAT_R8G8B8A8_UNORM_SRGB:
#if HAVE_SRGB
      return PIPE_FORMAT_R8G8B8A8_SRGB;
#else
      return PIPE_FORMAT_R8G8B8A8_UNORM;
#endif

   /* Gallium has no typeless formats.  We simply use an equivalent
    * typed format.
     */
   case DXGI_FORMAT_R8G8B8A8_TYPELESS:
      return PIPE_FORMAT_R8G8B8A8_UNORM;
   case DXGI_FORMAT_BC1_TYPELESS:
      return PIPE_FORMAT_DXT1_RGBA;
   case DXGI_FORMAT_BC2_TYPELESS:
      return PIPE_FORMAT_DXT3_RGBA;
   case DXGI_FORMAT_BC3_TYPELESS:
      return PIPE_FORMAT_DXT5_RGBA;
   case DXGI_FORMAT_R8_TYPELESS:
      return PIPE_FORMAT_R8_UNORM;
   case DXGI_FORMAT_R32G32B32A32_TYPELESS:
      return PIPE_FORMAT_R32G32B32A32_UNORM;
   case DXGI_FORMAT_R32G32B32_TYPELESS:
      return PIPE_FORMAT_R32G32B32_UNORM;
   case DXGI_FORMAT_R16G16B16A16_TYPELESS:
      return PIPE_FORMAT_R16G16B16A16_UNORM;
   case DXGI_FORMAT_R32G32_TYPELESS:
      return PIPE_FORMAT_R32G32_UNORM;
   case DXGI_FORMAT_R8G8_TYPELESS:
      return PIPE_FORMAT_R8G8_UNORM;
   case DXGI_FORMAT_R16_TYPELESS:
      return depth ? PIPE_FORMAT_Z16_UNORM : PIPE_FORMAT_R16_UNORM;
   case DXGI_FORMAT_R16G16_TYPELESS:
      return PIPE_FORMAT_R16G16_UNORM;
   case DXGI_FORMAT_R32_TYPELESS:
      return depth ? PIPE_FORMAT_Z32_FLOAT : PIPE_FORMAT_R32_FLOAT;
   case DXGI_FORMAT_R24G8_TYPELESS:
      return PIPE_FORMAT_Z24_UNORM_S8_UINT;
   case DXGI_FORMAT_R24_UNORM_X8_TYPELESS:
      return PIPE_FORMAT_Z24X8_UNORM;
   case DXGI_FORMAT_X24_TYPELESS_G8_UINT:
      return PIPE_FORMAT_Z24_UNORM_S8_UINT;

   case DXGI_FORMAT_R8G8_B8G8_UNORM:
      return PIPE_FORMAT_R8G8_B8G8_UNORM;
   case DXGI_FORMAT_G8R8_G8B8_UNORM:
      return PIPE_FORMAT_G8R8_G8B8_UNORM;

   case DXGI_FORMAT_R16G16B16A16_FLOAT:
      return PIPE_FORMAT_R16G16B16A16_FLOAT;
   case DXGI_FORMAT_R16G16_FLOAT:
      return PIPE_FORMAT_R16G16_FLOAT;
   case DXGI_FORMAT_R16_FLOAT:
      return PIPE_FORMAT_R16_FLOAT;
   case DXGI_FORMAT_R9G9B9E5_SHAREDEXP:
      return PIPE_FORMAT_R9G9B9E5_FLOAT;
   case DXGI_FORMAT_R1_UNORM:
      return PIPE_FORMAT_R1_UNORM;

   case DXGI_FORMAT_D32_FLOAT_S8X24_UINT:
   case DXGI_FORMAT_R32G8X24_TYPELESS:
   case DXGI_FORMAT_R32_FLOAT_X8X24_TYPELESS:
   case DXGI_FORMAT_X32_TYPELESS_G8X24_UINT:
      return PIPE_FORMAT_Z32_FLOAT_S8X24_UINT;

   case DXGI_FORMAT_R10G10B10A2_TYPELESS:
   case DXGI_FORMAT_R10G10B10A2_UNORM:
       return PIPE_FORMAT_R10G10B10A2_UNORM;
   case DXGI_FORMAT_R10G10B10A2_UINT:
      return PIPE_FORMAT_R10G10B10A2_UINT;
   case DXGI_FORMAT_R11G11B10_FLOAT:
      return PIPE_FORMAT_R11G11B10_FLOAT;

   case DXGI_FORMAT_BC4_TYPELESS:
   case DXGI_FORMAT_BC4_UNORM:
      return PIPE_FORMAT_RGTC1_UNORM;
   case DXGI_FORMAT_BC4_SNORM:
      return PIPE_FORMAT_RGTC1_SNORM;

   case DXGI_FORMAT_BC5_TYPELESS:
   case DXGI_FORMAT_BC5_UNORM:
      return PIPE_FORMAT_RGTC2_UNORM;
   case DXGI_FORMAT_BC5_SNORM:
      return PIPE_FORMAT_RGTC2_SNORM;

   case DXGI_FORMAT_B5G6R5_UNORM:
      return PIPE_FORMAT_B5G6R5_UNORM;
   case DXGI_FORMAT_B5G5R5A1_UNORM:
      return PIPE_FORMAT_B5G5R5A1_UNORM;
   case DXGI_FORMAT_B8G8R8A8_UNORM:
      return PIPE_FORMAT_B8G8R8A8_UNORM;
   case DXGI_FORMAT_B8G8R8X8_UNORM:
      return PIPE_FORMAT_B8G8R8X8_UNORM;
   case DXGI_FORMAT_B8G8R8A8_TYPELESS:
      return PIPE_FORMAT_B8G8R8A8_UNORM;
   case DXGI_FORMAT_B8G8R8A8_UNORM_SRGB:
#if HAVE_SRGB
      return PIPE_FORMAT_B8G8R8A8_SRGB;
#else
      return PIPE_FORMAT_B8G8R8A8_UNORM;
#endif
   case DXGI_FORMAT_B8G8R8X8_TYPELESS:
      return PIPE_FORMAT_B8G8R8X8_UNORM;
   case DXGI_FORMAT_B8G8R8X8_UNORM_SRGB:
#if HAVE_SRGB
      return PIPE_FORMAT_B8G8R8X8_SRGB;
#else
      return PIPE_FORMAT_B8G8R8X8_UNORM;
#endif

   case DXGI_FORMAT_R10G10B10_XR_BIAS_A2_UNORM:
      return PIPE_FORMAT_R10G10B10A2_UNORM;

   default:
      DebugPrintf("%s: unsupported format %s\n",
                  __func__, FormatToName(Format));
      return PIPE_FORMAT_NONE;
   }
}



#define CASE(fmt) case fmt: return #fmt

const char *
FormatToName(DXGI_FORMAT Format)
{
   switch (Format) {
   CASE(DXGI_FORMAT_UNKNOWN);
   CASE(DXGI_FORMAT_R32G32B32A32_TYPELESS);
   CASE(DXGI_FORMAT_R32G32B32A32_FLOAT);
   CASE(DXGI_FORMAT_R32G32B32A32_UINT);
   CASE(DXGI_FORMAT_R32G32B32A32_SINT);
   CASE(DXGI_FORMAT_R32G32B32_TYPELESS);
   CASE(DXGI_FORMAT_R32G32B32_FLOAT);
   CASE(DXGI_FORMAT_R32G32B32_UINT);
   CASE(DXGI_FORMAT_R32G32B32_SINT);
   CASE(DXGI_FORMAT_R16G16B16A16_TYPELESS);
   CASE(DXGI_FORMAT_R16G16B16A16_FLOAT);
   CASE(DXGI_FORMAT_R16G16B16A16_UNORM);
   CASE(DXGI_FORMAT_R16G16B16A16_UINT);
   CASE(DXGI_FORMAT_R16G16B16A16_SNORM);
   CASE(DXGI_FORMAT_R16G16B16A16_SINT);
   CASE(DXGI_FORMAT_R32G32_TYPELESS);
   CASE(DXGI_FORMAT_R32G32_FLOAT);
   CASE(DXGI_FORMAT_R32G32_UINT);
   CASE(DXGI_FORMAT_R32G32_SINT);
   CASE(DXGI_FORMAT_R32G8X24_TYPELESS);
   CASE(DXGI_FORMAT_D32_FLOAT_S8X24_UINT);
   CASE(DXGI_FORMAT_R32_FLOAT_X8X24_TYPELESS);
   CASE(DXGI_FORMAT_X32_TYPELESS_G8X24_UINT);
   CASE(DXGI_FORMAT_R10G10B10A2_TYPELESS);
   CASE(DXGI_FORMAT_R10G10B10A2_UNORM);
   CASE(DXGI_FORMAT_R10G10B10A2_UINT);
   CASE(DXGI_FORMAT_R11G11B10_FLOAT);
   CASE(DXGI_FORMAT_R8G8B8A8_TYPELESS);
   CASE(DXGI_FORMAT_R8G8B8A8_UNORM);
   CASE(DXGI_FORMAT_R8G8B8A8_UNORM_SRGB);
   CASE(DXGI_FORMAT_R8G8B8A8_UINT);
   CASE(DXGI_FORMAT_R8G8B8A8_SNORM);
   CASE(DXGI_FORMAT_R8G8B8A8_SINT);
   CASE(DXGI_FORMAT_R16G16_TYPELESS);
   CASE(DXGI_FORMAT_R16G16_FLOAT);
   CASE(DXGI_FORMAT_R16G16_UNORM);
   CASE(DXGI_FORMAT_R16G16_UINT);
   CASE(DXGI_FORMAT_R16G16_SNORM);
   CASE(DXGI_FORMAT_R16G16_SINT);
   CASE(DXGI_FORMAT_R32_TYPELESS);
   CASE(DXGI_FORMAT_D32_FLOAT);
   CASE(DXGI_FORMAT_R32_FLOAT);
   CASE(DXGI_FORMAT_R32_UINT);
   CASE(DXGI_FORMAT_R32_SINT);
   CASE(DXGI_FORMAT_R24G8_TYPELESS);
   CASE(DXGI_FORMAT_D24_UNORM_S8_UINT);
   CASE(DXGI_FORMAT_R24_UNORM_X8_TYPELESS);
   CASE(DXGI_FORMAT_X24_TYPELESS_G8_UINT);
   CASE(DXGI_FORMAT_R8G8_TYPELESS);
   CASE(DXGI_FORMAT_R8G8_UNORM);
   CASE(DXGI_FORMAT_R8G8_UINT);
   CASE(DXGI_FORMAT_R8G8_SNORM);
   CASE(DXGI_FORMAT_R8G8_SINT);
   CASE(DXGI_FORMAT_R16_TYPELESS);
   CASE(DXGI_FORMAT_R16_FLOAT);
   CASE(DXGI_FORMAT_D16_UNORM);
   CASE(DXGI_FORMAT_R16_UNORM);
   CASE(DXGI_FORMAT_R16_UINT);
   CASE(DXGI_FORMAT_R16_SNORM);
   CASE(DXGI_FORMAT_R16_SINT);
   CASE(DXGI_FORMAT_R8_TYPELESS);
   CASE(DXGI_FORMAT_R8_UNORM);
   CASE(DXGI_FORMAT_R8_UINT);
   CASE(DXGI_FORMAT_R8_SNORM);
   CASE(DXGI_FORMAT_R8_SINT);
   CASE(DXGI_FORMAT_A8_UNORM);
   CASE(DXGI_FORMAT_R1_UNORM);
   CASE(DXGI_FORMAT_R9G9B9E5_SHAREDEXP);
   CASE(DXGI_FORMAT_R8G8_B8G8_UNORM);
   CASE(DXGI_FORMAT_G8R8_G8B8_UNORM);
   CASE(DXGI_FORMAT_BC1_TYPELESS);
   CASE(DXGI_FORMAT_BC1_UNORM);
   CASE(DXGI_FORMAT_BC1_UNORM_SRGB);
   CASE(DXGI_FORMAT_BC2_TYPELESS);
   CASE(DXGI_FORMAT_BC2_UNORM);
   CASE(DXGI_FORMAT_BC2_UNORM_SRGB);
   CASE(DXGI_FORMAT_BC3_TYPELESS);
   CASE(DXGI_FORMAT_BC3_UNORM);
   CASE(DXGI_FORMAT_BC3_UNORM_SRGB);
   CASE(DXGI_FORMAT_BC4_TYPELESS);
   CASE(DXGI_FORMAT_BC4_UNORM);
   CASE(DXGI_FORMAT_BC4_SNORM);
   CASE(DXGI_FORMAT_BC5_TYPELESS);
   CASE(DXGI_FORMAT_BC5_UNORM);
   CASE(DXGI_FORMAT_BC5_SNORM);
   CASE(DXGI_FORMAT_B5G6R5_UNORM);
   CASE(DXGI_FORMAT_B5G5R5A1_UNORM);
   CASE(DXGI_FORMAT_B8G8R8A8_UNORM);
   CASE(DXGI_FORMAT_B8G8R8X8_UNORM);
   CASE(DXGI_FORMAT_R10G10B10_XR_BIAS_A2_UNORM);
   CASE(DXGI_FORMAT_B8G8R8A8_TYPELESS);
   CASE(DXGI_FORMAT_B8G8R8A8_UNORM_SRGB);
   CASE(DXGI_FORMAT_B8G8R8X8_TYPELESS);
   CASE(DXGI_FORMAT_B8G8R8X8_UNORM_SRGB);
   CASE(DXGI_FORMAT_BC6H_TYPELESS);
   CASE(DXGI_FORMAT_BC6H_UF16);
   CASE(DXGI_FORMAT_BC6H_SF16);
   CASE(DXGI_FORMAT_BC7_TYPELESS);
   CASE(DXGI_FORMAT_BC7_UNORM);
   CASE(DXGI_FORMAT_BC7_UNORM_SRGB);
   default:
      return "???";
   }
}
