/*
 * Copyright (c) 2019-2020 Dmitry Belyavskiy <beldmit@gmail.com>
 *
 * Contents licensed under the terms of the OpenSSL license
 * See https://www.openssl.org/source/license.html for details
 */
#ifdef _MSC_VER
# pragma warning(push, 3)
# include <openssl/applink.c>
# pragma warning(pop)
# include <Winsock2.h>
# include <stdlib.h>
#else
# include <arpa/inet.h>
#endif
#include <string.h>
#include <stdio.h>
#include <string.h>
#include <openssl/evp.h>
#include <openssl/hmac.h>
#include <openssl/obj_mac.h>
#include "gost_lcl.h"
#include "gost_gost2015.h"
#include "e_gost_err.h"
#include "gost_grasshopper_cipher.h"

#define T(e) \
    if (!(e)) { \
        ERR_print_errors_fp(stderr); \
        OpenSSLDie(__FILE__, __LINE__, #e); \
    }

static void hexdump(FILE *f, const char *title, const unsigned char *s, int l)
{
    int n = 0;

    fprintf(f, "%s", title);
    for (; n < l; ++n) {
        if ((n % 16) == 0)
            fprintf(f, "\n%04x", n);
        fprintf(f, " %02x", s[n]);
    }
    fprintf(f, "\n");
}

static int expect_eq(const char *test_name, int ret, const unsigned char *result,
                     const unsigned char *expected, size_t len)
{
    if (ret <= 0) {
        ERR_print_errors_fp(stderr);
        return 1;
    } else {
        hexdump(stdout, test_name, result, len);
        if (memcmp(result, expected, len) != 0) {
            fprintf(stdout, "ERROR! %s test failed\n", test_name);
            return 2;
        }
    }
    return 0;
}

static int initialize_openssl(ENGINE **eng)
{
#ifdef _MSC_VER
    _putenv_s("OPENSSL_ENGINES", ENGINE_DIR);
#else
    setenv("OPENSSL_ENGINES", ENGINE_DIR, 0);
#endif
    OPENSSL_add_all_algorithms_conf();
    ERR_load_crypto_strings();
    T(*eng = ENGINE_by_id("gost"));
    T(ENGINE_init(*eng));
    T(ENGINE_set_default(*eng, ENGINE_METHOD_ALL));

   /*
    * The GOST_NID_JOB structs statically linked into the test start uninitialized,
    * so we must assign their NIDs manually.
    */
    kuznyechik_mgm_NID.callback(OBJ_sn2nid(SN_kuznyechik_mgm));
    magma_mgm_NID.callback(OBJ_sn2nid(SN_magma_mgm));
    return 0;
}

static void cleanup_openssl(ENGINE *eng)
{
    ENGINE_finish(eng);
    ENGINE_free(eng);
}

int main(void)
{
    const unsigned char shared_key[] = {
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF
    };

    const unsigned char magma_key[] = {
        0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
        0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
        0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
    };

    unsigned char mac_magma_key[] = {
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
    };

    const unsigned char magma_iv[] = { 0x67, 0xBE, 0xD6, 0x54 };

    const unsigned char magma_export[] = {
        0xCF, 0xD5, 0xA1, 0x2D, 0x5B, 0x81, 0xB6, 0xE1,
        0xE9, 0x9C, 0x91, 0x6D, 0x07, 0x90, 0x0C, 0x6A,
        0xC1, 0x27, 0x03, 0xFB, 0x3A, 0xBD, 0xED, 0x55,
        0x56, 0x7B, 0xF3, 0x74, 0x2C, 0x89, 0x9C, 0x75,
        0x5D, 0xAF, 0xE7, 0xB4, 0x2E, 0x3A, 0x8B, 0xD9
    };

    unsigned char kdftree_key[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
    };

    unsigned char kdf_label[] = { 0x26, 0xBD, 0xB8, 0x78 };
    unsigned char kdf_seed[] = {
        0xAF, 0x21, 0x43, 0x41, 0x45, 0x65, 0x63, 0x78
    };
    const unsigned char kdf_etalon[] = {
        0x22, 0xB6, 0x83, 0x78, 0x45, 0xC6, 0xBE, 0xF6,
        0x5E, 0xA7, 0x16, 0x72, 0xB2, 0x65, 0x83, 0x10,
        0x86, 0xD3, 0xC7, 0x6A, 0xEB, 0xE6, 0xDA, 0xE9,
        0x1C, 0xAD, 0x51, 0xD8, 0x3F, 0x79, 0xD1, 0x6B,
        0x07, 0x4C, 0x93, 0x30, 0x59, 0x9D, 0x7F, 0x8D,
        0x71, 0x2F, 0xCA, 0x54, 0x39, 0x2F, 0x4D, 0xDD,
        0xE9, 0x37, 0x51, 0x20, 0x6B, 0x35, 0x84, 0xC8,
        0xF4, 0x3F, 0x9E, 0x6D, 0xC5, 0x15, 0x31, 0xF9
    };

    const unsigned char tlstree_gh_etalon[] = {
        0x50, 0x76, 0x42, 0xd9, 0x58, 0xc5, 0x20, 0xc6,
        0xd7, 0xee, 0xf5, 0xca, 0x8a, 0x53, 0x16, 0xd4,
        0xf3, 0x4b, 0x85, 0x5d, 0x2d, 0xd4, 0xbc, 0xbf,
        0x4e, 0x5b, 0xf0, 0xff, 0x64, 0x1a, 0x19, 0xff,
    };

    const unsigned char kroot_kuzn_s[] = {
        0x58, 0x16, 0x88, 0xD7, 0x6E, 0xFE, 0x12, 0x2B,
        0xB5, 0x5F, 0x62, 0xB3, 0x8E, 0xF0, 0x1B, 0xCC,
        0x8C, 0x88, 0xDB, 0x83, 0xE9, 0xEA, 0x4D, 0x55,
        0xD3, 0x89, 0x8C, 0x53, 0x72, 0x1F, 0xC3, 0x84
    };

    const unsigned char tlstree_kuzn_s_etalon[] = {
        0xE1, 0xC5, 0x9B, 0x41, 0x69, 0xD8, 0x96, 0x10,
        0x7F, 0x78, 0x45, 0x68, 0x93, 0xA3, 0x75, 0x1E,
        0x15, 0x73, 0x54, 0x3D, 0xAD, 0x8C, 0xB7, 0x40,
        0x69, 0xE6, 0x81, 0x4A, 0x51, 0x3B, 0xBB, 0x1C
    };

    unsigned char kroot_magma_l[] = {
        0xDF, 0x66, 0x60, 0x1E, 0xDD, 0xD6, 0x4E, 0x96,
        0x1D, 0xFC, 0x7D, 0xD0, 0x21, 0x2E, 0xF2, 0x25,
        0xC0, 0x05, 0x33, 0xE6, 0xDA, 0xA4, 0xAD, 0x24,
        0x18, 0x5E, 0xBE, 0xB2, 0x24, 0xB5, 0x46, 0xB8
    };

    unsigned char tlstree_magma_l_etalon[] = {
        0xBD, 0x00, 0x9F, 0xFC, 0x04, 0xA0, 0x52, 0x9E,
        0x60, 0x78, 0xEB, 0xA5, 0xA0, 0x7A, 0xDE, 0x74,
        0x93, 0x7F, 0xF3, 0xA1, 0xAB, 0x75, 0xF7, 0xAE,
        0x05, 0x19, 0x04, 0x78, 0x51, 0x9B, 0x6D, 0xF3
    };

    unsigned char buf[32 + 16];
    int ret = 0, err = 0;
    int outlen = 40;
    unsigned char kdf_result[64];

    unsigned char kroot[32];
    unsigned char tlsseq[8];
    unsigned char out[32];

    ENGINE *eng;
    if (initialize_openssl(&eng) != 0) {
        return 1;
    }

    memset(buf, 0, sizeof(buf));
    memset(kroot, 0xFF, 32);
    memset(tlsseq, 0, 8);
    tlsseq[7] = 63;
    memset(out, 0, 32);

    ret = gost_kexp15(shared_key, 32,
                      NID_magma_ctr, magma_key,
                      NID_magma_mac, mac_magma_key, magma_iv, 4, buf, &outlen);
    err = expect_eq("Magma key export", ret, buf, magma_export, 40);
    if (err)
        goto cleanup;

    ret = gost_kimp15(magma_export, 40,
                      NID_magma_ctr, magma_key,
                      NID_magma_mac, mac_magma_key, magma_iv, 4, buf);
    err = expect_eq("Magma key import", ret, buf, shared_key, 32);
    if (err)
        goto cleanup;

    ret = gost_kdftree2012_256(kdf_result, 64, kdftree_key, 32, kdf_label, 4,
                              kdf_seed, 8, 1);
    err = expect_eq("KDF TREE", ret, kdf_result, kdf_etalon, 64);
    if (err)
        goto cleanup;

    ret = gost_tlstree(NID_grasshopper_cbc, kroot, out, tlsseq, TLSTREE_MODE_NONE);
    err = expect_eq("Gost TLSTREE_MODE_NONE - grasshopper", ret, out, tlstree_gh_etalon, 32);
    if (err)
        goto cleanup;

    tlsseq[7] = 7;
    ret = gost_tlstree(OBJ_sn2nid(SN_kuznyechik_mgm), kroot_kuzn_s, out, tlsseq, TLSTREE_MODE_S);
    err = expect_eq("Gost TLSTREE_MODE_S - grasshopper", ret, out, tlstree_kuzn_s_etalon, 32);
    if (err)
        goto cleanup;

    tlsseq[7] = 7;
    ret = gost_tlstree(OBJ_sn2nid(SN_magma_mgm), kroot_magma_l, out, tlsseq, TLSTREE_MODE_L);
    err = expect_eq("Gost TLSTREE_MODE_L - magma", ret, out, tlstree_magma_l_etalon, 32);
    if (err)
        goto cleanup;

cleanup:
    cleanup_openssl(eng);
    return err;
}